--agora é o evaluator (interpretador)
--ele é responsável por percorrer a árvore de nodes (AST) criada pelo parser
--e dar significado real a ela (executar)

--criar valores, declarar variáveis, executar blocos, comparar, somar, etc
--é com base no "kind" do node ele decido oque fazer, como declara variavel ou criar um entity(a representação do valor/dado da linguagem)

--entity pode ser qualquer coisa, como desde um booleano até um erro ou coisas mais complexa como um array ou declaração de classes

--o evaluator é a parte mais importante da linguagem
--ele é onde tudo deixa de ser “estrutura” e passa a ser “comportamento”

--lexer e parser só organizam
--evaluator é quem dá vida

local Scope = require("scope")
local Entity = require("entity")

---@class Evaluator
local Evaluator = {}
---aqui é onde será interpretado cada node criado pelo parser, onde tudo ganha vida
---@param node Node
---@param scope ?Scope
function Evaluator:process(node, scope)
    local kind, value = node.kind, node.value

    if kind == "LiteralNil" then
        return Entity.new("Nil") --aqui cria o nulo
    elseif kind == "LiteralBoolean" then --aqui cria o booleano
        return Entity.new("Boolean", value)
    elseif kind == "LiteralNumber" then --aqui cria o número
        return Entity.new("Number", value)
    elseif kind == "LiteralString" then --aqui cria o texto/string
        return Entity.new("String", value)
    elseif kind == "LiteralTable" then
        local elements = {}
        for _, content in ipairs(value) do
            local result = self:process(content, scope)
            if result:check("Error") then
                return result
            end

            table.insert(elements, result)
        end
        return Entity.new("Table", elements)
    elseif kind == "LiteralCallable" then --aqui cria a função
        return Entity.new("Function", {parameters = value.parameters, block = value.block, scope = scope})
    elseif kind == "DeclareStatement" then
        --processa o valor da variavel
        local expression = self:process(value.value, scope)
        if expression:check("Error") then
            return expression
        end

        --checagem para saber se está declarando uma variavel com um nome que já exite no mesmo escopo
        if not scope:makeNewDeclaration(value.name, expression) then
            --caso tenha, emite o erro
            return Entity.new("Error", "tentativa de declarar duas variaveis com o mesmo nome " .. value.name .. " no mesmo escopo")
        end
    elseif kind == "DeferStatement" then
        --declara um novo defer(adiamente de evento)
        scope:declareDefer(value)
    elseif kind == "ForStatement" then
        local environment = Scope.new(scope)

        self:process(value.variable, environment)

        while true do
            local current = environment:getDeclaration(value.variable.value.name)

            if current:check("Error") then
                return current
            elseif not current:check("Number") then
                return Entity.new("Error", "esperado um número na variavel '" .. value.variable.value.name .. "'")
            end

            local maximum = self:process(value.maximum, environment)
            if maximum:check("Error") then
                return maximum
            elseif not maximum:check("Number") then
                return Entity.new("Error", "esperado um número como máximo de passos que será dado")
            end

            if current.value < maximum.value then
                local result = self:process(value.block, environment)
                if result:check("Error") then
                    return result
                end

                local step = self:process(value.step, environment)
                if step:check("Error") then
                    return step
                elseif not step:check("Number") then
                    return Entity.new("Error", "esperado um número como número de passos que será dado")
                end

                environment:setDeclaration(value.variable.value.name, Entity.new("Number", current.value + step.value))
            else
                break
            end
        end
    elseif kind == "WhileStatement" then
        while true do
            local expression = self:process(value.expression, scope)
            if expression:check("Error") then
                return expression
            elseif expression:check("Boolean") then
                if expression.value then
                    self:process(value.block, scope)
                else
                    break
                end
            else
                return Entity.new("Error", "loop while espera um booleano resultado de expressão")
            end
        end
    elseif kind == "IfStatement" then
        local expression = self:process(value.expression, scope)

        --checa se a expressão deu erro, case tenha dado propaga esse erro até a raiz da execução
        if expression:check("Error") then
            return expression
        --caso não sejá um erro, checa se é um booleano
        --também pode criar outros comportamentos
        --usar outros tipos de dado como o proprio lua faz com o "nil" que é o mesmo que false
        elseif expression:check("Boolean") then
            if expression.value then
                --caso a condição atenda, executa o bloco do if
                return self:process(value.block, scope)
            elseif node.value.unless then --caso a condição não atenda e possua o else, executa o else
                return self:process(value.unless, scope)
            end
        else
            --aqui está emitindo um erro sobre usar dados não booleano como expressão de um if
            return Entity.new("Error", "experado resultado booleano de expressão no if")
        end
    elseif kind == "BlockStatement" then
        --cria um escopo/ambiente temporaria para o bloco
        local environment = Scope.new(scope)

        --percorre o conteudo do bloco passando por cada node e o processa
        for _, content in ipairs(value) do
            local result = self:process(content, environment)

            --caso o conteudo de erro, é propagado
            if result:check("Error") then
                return result
            end
        end

        --execução de defer(execuções adiadas)

        local defer = environment:popDefer()
        while defer do
            local result = self:process(defer, environment)

            --caso o conteudo de erro, é propagado
            if result:check("Error") then
                return result
            end

            defer = environment:popDefer()
        end
    elseif kind == "AssignmentStatement" then
        local target = value.target
        while true do
            if target.kind == "Reference" then
                break
            elseif target.kind == "IndexAccess" then
            end
        end

        local expression = self:process(node.value.expression, scope)
        if expression:check("Error") then
            return expression
        end

        if not scope:setDeclaration(target.value, expression) then
            return Entity.new("Error", "tentativa de atribuir valor a uma variavel que não exite '" .. node.value.name .. "'")
        end
    elseif kind == "StatementExpression" then
        --aqui é processado as expressões como "print()", "1 + 1"
        --expressão que não possuem um alvo para recebela
        return self:process(value, scope)
    elseif kind == "Reference" then
        --faz a busca da variavel pelo nome
        local reference = scope:getDeclaration(value)
        if not reference then --checa se a variavel não exite e emite um erro
            return Entity.new("Error", "não foi encontrada a variavel com o nome '" .. node.value .. "'")
        end
        --caso a variavel exite, processa o valor que ela possui
        return reference
    elseif kind == "IndexAccess" then
        local index = self:process(value.index, scope)
        if index:check("Error") then
            return false
        elseif index:check("Number") then
            local base = self:process(value.base, scope)
            if base:check("Error") then
                return base
            elseif base:check("Table") then
                if index.value > #base.value then
                    return Entity.new("Error", "índice fora do alcance")
                end
                return base.value[index.value]
            else
                return Entity.new("Error", "não suportado acesso a índice em um tipo que não sejá tabela")
            end
        else
            return Entity.new("Error", "esperado número em acesso de índice")
        end
    elseif kind == "CallAccess" then
        --aqui fica os argumentos processada da função que está sendo chamada
        local arguments = {}
        --esse loop processa a lista de argumentos e guarda o resultado em "arguments"(acima)
        for _, operand in ipairs(value.arguments) do
            local result = self:process(operand, scope)
            if result:check("Error") then --propaga erro caso sejá um
                return result
            end
            table.insert(arguments, result) --guarda o resultado do processo do argumento em "arguments"(acima)
        end

        local callable = self:process(value.base, scope)
        if callable:check("Error") then
            return callable
        elseif callable:check("Function") then
            local environment = Scope.new(callable.value.scope)

            for index, name in ipairs(callable.value.parameters) do
                environment:makeNewDeclaration(name, arguments[index])
            end

            local result = self:process(callable.value.block, environment)
            return result
        else
            return Entity.new("Error", "tentativa de usar dado como chamada de função")
    end
    elseif kind == "Callable" then
        --aqui fica os argumentos processada da função que está sendo chamada
        local arguments = {}
        --esse loop processa a lista de argumentos e guarda o resultado em "arguments"(acima)
        for _, operand in ipairs(value.arguments) do
            local result = self:process(operand, scope)
            if result:check("Error") then --propaga erro caso sejá um
                return result
            end
            table.insert(arguments, result) --guarda o resultado do processo do argumento em "arguments"(acima)
        end

        if value.name == "print" then --aqui está sendo interpredado o print usando o "io.write" para escrever no terminal
            for _, operand in ipairs(arguments) do
                io.write(operand:format())
            end
            io.write("\n")
        else
            local callable = scope:getDeclaration(value.name)
            if not callable then
                return Entity.new("Error", "não foi encontrado função '" .. value.name .. "'")
            elseif not callable:check("Function") then
                return Entity.new("Error", "tentativa de usar dado como função")
            end

            if callable then
                local environment = Scope.new(callable.value.scope)

                for index, name in ipairs(callable.value.parameters) do
                    environment:makeNewDeclaration(name, arguments[index])
                end

                local result = self:process(callable.value.block, environment)
                return result
            end
        end
    elseif kind == "ArithmeticExpression" then
        --processa o lado esquerdo da expressão
        --exemplo 1 + ...
        --        ^
        local left = self:process(node.value.left, scope)
        if left:check("Error") then --checagem de erro, caso sejá um, o propaga
            return left
        end

        --processa o lado direito da expressão
        --exemplo ... + 1
        --              ^
        local right = self:process(node.value.right, scope)
        if right:check("Error") then --checagem de erro, caso sejá um, o propaga
            return right
        end

        if left:check("Number") and right:check("Number") then --checagem se a expressão possui o tipo correto
            if node.value.operator == '+' then
                return Entity.new("Number", left.value + right.value)
            elseif node.value.operator == '-' then
                return Entity.new("Number", left.value - right.value)
            elseif node.value.operator == '*' then
                return Entity.new("Number", left.value * right.value)
            elseif node.value.operator == '/' then
                return Entity.new("Number", left.value / right.value)
            end
        end
        --caso a expressão possua um tipo que não sejá um número
        return Entity.new("Error", "experado 2 números em expressão aritimeticas")
    elseif kind == "ComparisionExpression" then
        --aqui é bem parecido com a expressão aritimetica

        --processa o lado esquerdo da expressão
        --exemplo 1 == ...
        --        ^
        local left = self:process(node.value.left, scope)
        if left:check("Error") then --checagem de erro, caso sejá um, o propaga
            return left
        end

        --processa o lado esquerdo da expressão
        --exemplo ... == 1
        --               ^
        local right = self:process(node.value.right, scope)
        if right:check("Error") then --checagem de erro, caso sejá um, o propaga
            return right
        end

        if not left:check(right.kind) then
            --caso os dois lados da expressão sejá tipos diferentes, retorna um false
            return Entity.new("Boolean", false)
        end

        if left:check("Nil") then
            if node.value.operator == '==' then --verifica se está usando o operador ==
                return Entity.new("Boolean", true)
            end
            --caso o operador não sejá == propaga um erro
            return Entity.new("Error", "não é suportado outro operador que não sejá 'igual' entre nil")
        elseif left:check("Boolean") then --verifica se os dois lados são booleanos
            if node.value.operator == '==' then --verifica se está usando o operador ==
                return Entity.new("Boolean", left.value == right.value)
            end
            --caso o operador não sejá == propaga um erro
            return Entity.new("Error", "não é suportado outro operador que não sejá 'igual' entre booleanos")
        elseif left:check("Number") then --verifica se os dois lados são números
            local operator = value.operator

            --aqui será interpretado os operadores ==, >, >=, <, <= pois todos são suportados em entre números
            if operator == '==' then
                return Entity.new("Boolean", left.value == right.value)
            elseif operator == '>' then
                return Entity.new("Boolean", left.value > right.value)
            elseif operator == '>=' then
                return Entity.new("Boolean", left.value >= right.value)
            elseif operator == '<' then
                return Entity.new("Boolean", left.value < right.value)
            elseif operator == '<=' then
                return Entity.new("Boolean", left.value <= right.value)
            end
        elseif left:check("String") then --verifica se os dois lados são textos/strings
            if node.value.operator == '==' then --verifica se está usando o operador ==
                return Entity.new("Boolean", left.value == right.value)
            end
            --caso o operador não sejá == propaga um erro, pois o unico operador possivel é o ==
            --claro que pode adicionar outros operadores
            --nada impede de fazer "cachorro" > "gato" e isso pode seila, comparar tamanho!
            return Entity.new("Error", "não é suportado outro operador que não sejá 'igual' entre texto/strings")
        end
    elseif kind == "LogicalExpression" then
        --aqui é bem parecido com a expressão aritimetica e de comparação

        --processa o lado esquerdo da expressão
        --exemplo true and ...
        --        ^
        local left = self:process(node.value.left, scope)
        if left:check("Error") then
            return left
        end

        --processa o lado esquerdo da expressão
        --exemplo ... and true
        --                 ^
        local right = self:process(node.value.right, scope)
        if right:check("Error") then
            return right
        end

        if left:check("Boolean") and right:check("Boolean") then --verifica se os lados são booleanos
            if value.operator == "&&" then --caso a expressão sejá "and"
                return Entity.new("Boolean", left.value and right.value)
            else --caso a expressão sejá "or"
                return Entity.new("Boolean", left.value or right.value)
            end
        end
        --caso a expressão tenha um tipo não suportado, só suporta booleanos
        return Entity.new("Error", "não suportado expressão logica entre tipos que não sejá booleanos")
    elseif kind == "UnaryExpression" then
        local expression = self:process(node.value.expression, scope)
        if expression:check("Error") then
            return expression
        end

        --aqui é onde inverte 1 para -1 ou -(-1) para +1 quando usado o -
        --também inverte true para false ou false para true quando usado o not

        if value.operator == '-' then
            if expression:check("Number") then
                return Entity.new("Number", -expression.value)
            end
            return Entity.new("Error", "unario de negativação só é suportado em número")
        else
            --aqui é feito uma checagem para saber se a expressão dentro de "not" é um booleano
            if expression:check("Boolean") then
                return Entity.new("Boolean", not expression.value)
            end
            return Entity.new("Error", "unario de negação só é suportado em booleano")
        end
    elseif kind == "GroupedExpression" then
        --processa a expressão entre (...)
        --isso é importante para dar foco a uma expressão, caso tenha uma expressão com a precedencia menor
        --exemplo
        --1 + 2 * 3 esse expressão é executada 2 * 3 = 6 depois será feito 1 + 6 = 7
        --isso acontece porque * é mais "forte" que +
        --se quizer fazer 1 + 2 primeiro, basta fazer (1 + 2) * 3
        --será executado como 1 + 2 = 3 depois 3 * 3 = 9
        --foi como se tivesse dado mais "poder/força" a 1 + 2 quando foi usado (), deu foco para a +
        return self:process(node.value)
    end
    return Entity.new("Void")
end
return Evaluator